// Product management functions for Direct'sell marketplace

// Initialize products module
function initProducts() {
    loadAllProducts();
    setupProductEventListeners();
}

// Set up event listeners for product-related elements
function setupProductEventListeners() {
    // Product form submission
    const productForm = document.getElementById('product-form');
    if (productForm) {
        productForm.addEventListener('submit', handleProductSubmission);
    }

    // Image upload
    const uploadPlaceholder = document.getElementById('upload-placeholder');
    if (uploadPlaceholder) {
        uploadPlaceholder.addEventListener('click', () => {
            document.getElementById('product-images').click();
        });

        document.getElementById('product-images').addEventListener('change', handleImageUpload);
    }

    // Search functionality
    const searchInput = document.getElementById('search-input');
    const searchBtn = document.getElementById('search-btn');
    if (searchInput && searchBtn) {
        const debouncedSearch = debounce(performSearch, 300);
        searchInput.addEventListener('input', debouncedSearch);
        searchBtn.addEventListener('click', () => {
            performSearch();
        });
    }

    // Filter functionality
    const categoryFilter = document.getElementById('category-filter');
    const locationFilter = document.getElementById('location-filter');
    const conditionFilter = document.getElementById('condition-filter');
    
    if (categoryFilter) categoryFilter.addEventListener('change', applyFilters);
    if (locationFilter) locationFilter.addEventListener('change', applyFilters);
    if (conditionFilter) conditionFilter.addEventListener('change', applyFilters);

    // Contact buttons on product detail page
    const contactSellerBtn = document.getElementById('contact-seller-btn');
    const whatsappBtn = document.getElementById('whatsapp-btn');
    const callBtn = document.getElementById('call-btn');
    const markSoldBtn = document.getElementById('mark-sold-btn');
    const deleteProductBtn = document.getElementById('delete-product-btn');

    if (contactSellerBtn) contactSellerBtn.addEventListener('click', handleContactSeller);
    if (whatsappBtn) whatsappBtn.addEventListener('click', handleWhatsAppContact);
    if (callBtn) callBtn.addEventListener('click', handleCallContact);
    if (markSoldBtn) markSoldBtn.addEventListener('click', handleMarkAsSold);
    if (deleteProductBtn) deleteProductBtn.addEventListener('click', handleDeleteProduct);
}

// Handle product form submission
function handleProductSubmission(e) {
    e.preventDefault();

    if (!requireAuth()) return;

    const currentUser = getCurrentUser();

    // Get form values
    const title = document.getElementById('product-title').value.trim();
    const price = parseFloat(document.getElementById('product-price').value);
    const category = document.getElementById('product-category').value;
    const condition = document.getElementById('product-condition').value;
    const location = document.getElementById('product-location').value.trim();
    const description = document.getElementById('product-description').value.trim();

    // Validation
    if (!title || !price || !category || !condition || !location || !description) {
        showNotification('Please fill in all required fields', 'error');
        return;
    }

    if (price <= 0) {
        showNotification('Price must be greater than 0', 'error');
        return;
    }

    if (title.length < 3) {
        showNotification('Product title must be at least 3 characters', 'error');
        return;
    }

    if (description.length < 10) {
        showNotification('Description must be at least 10 characters', 'error');
        return;
    }

    // Get uploaded images
    const imageInput = document.getElementById('product-images');
    const imageFiles = imageInput.files;
    
    let images = [];
    
    if (imageFiles.length > 0) {
        // Process uploaded images
        for (let i = 0; i < imageFiles.length; i++) {
            const file = imageFiles[i];
            if (file.type.match('image.*')) {
                // For now, we'll store the file object and process it
                const reader = new FileReader();
                reader.onload = function(e) {
                    images.push(e.target.result);
                    
                    // Only save product when all images are processed
                    if (images.length === imageFiles.length) {
                        saveProductToStorage(title, price, category, condition, location, description, images, currentUser);
                    }
                };
                reader.readAsDataURL(file);
            } else {
                showNotification('File ' + (i+1) + ' is not an image', 'error');
                return;
            }
        }
    } else {
        // If no images uploaded, use placeholder
        images = ['data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="400" height="300" viewBox="0 0 400 300"><rect width="400" height="300" fill="#f0f0f0"/><text x="200" y="150" font-family="Arial" font-size="20" fill="#999" text-anchor="middle" alignment-baseline="middle">No Image</text></svg>'];
        saveProductToStorage(title, price, category, condition, location, description, images, currentUser);
    }
}

// Helper function to save product after images are processed
function saveProductToStorage(title, price, category, condition, location, description, images, currentUser) {
    // Create product object
    const newProduct = {
        id: generateId(),
        title: title,
        price: price,
        category: category,
        condition: condition,
        location: location,
        description: description,
        images: images,
        sellerId: currentUser.id,
        sellerName: currentUser.name,
        isSold: false,
        datePosted: new Date().toISOString()
    };

    // Save product
    const products = loadProducts();
    products.push(newProduct);
    saveProducts(products);

    // Update user's posted products
    currentUser.postedProducts.push(newProduct.id);
    updateCurrentUser(currentUser);
    saveUser(currentUser);

    showNotification('Product posted successfully!', 'success');

    // Redirect to profile after a delay
    setTimeout(() => {
        window.location.href = 'profile.html';
    }, 1500);
}

// Handle image upload
function handleImageUpload(e) {
    const files = e.target.files;
    const previewContainer = document.getElementById('image-preview');

    if (files.length === 0) return;

    // Clear previous previews
    previewContainer.innerHTML = '';

    // Limit to 5 images
    const fileArray = Array.from(files).slice(0, 5);

    // Process each file
    fileArray.forEach(file => {
        if (!file.type.match('image.*')) {
            showNotification('Please select image files only', 'error');
            return;
        }

        // Create preview element
        const reader = new FileReader();
        reader.onload = function(e) {
            const img = document.createElement('img');
            img.src = e.target.result;
            img.className = 'preview-image';
            img.style.width = '80px';
            img.style.height = '80px';
            img.style.objectFit = 'cover';
            img.style.borderRadius = '5px';
            img.style.margin = '5px';
            previewContainer.appendChild(img);
        };
        reader.readAsDataURL(file);
    });
}

// Load all products on homepage
function loadAllProducts() {
    const products = loadProducts();
    
    // Sort by newest first
    const sortedProducts = products.sort((a, b) => new Date(b.datePosted) - new Date(a.datePosted));
    
    const container = document.getElementById('products-grid');
    if (!container) return;
    
    // Clear loading message
    container.innerHTML = '';
    
    if (sortedProducts.length === 0) {
        container.innerHTML = `
            <div class="no-products-message">
                <h3>No products available yet</h3>
                <p>Be the first to post a product!</p>
                <a href="post-product.html" class="btn btn-primary">Post Your Product</a>
            </div>
        `;
        return;
    }
    
    sortedProducts.forEach(product => {
        if (!product.isSold) { // Only show non-sold products on homepage
            const productCard = createProductCard(product);
            container.appendChild(productCard);
        }
    });
    
    // Setup contact button event listeners
    setupProductCardContactButtons();
}

// Setup event listeners for contact buttons on product cards
function setupProductCardContactButtons() {
    // This function ensures all contact buttons work properly
    // Event listeners are already attached in createProductCard function
    // This is just a placeholder for future enhancements if needed
}

// Create a product card element
function createProductCard(product, showOwnerOptions = false) {
    const card = document.createElement('div');
    card.className = 'product-card';
    
    // Add sold badge if product is sold
    if (product.isSold) {
        card.innerHTML += `<span class="badge sold">Sold</span>`;
    }
    
    // Use the first image from the product, fallback to placeholder if none
    const mainImage = product.images && product.images.length > 0 ? product.images[0] : 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="400" height="300" viewBox="0 0 400 300"><rect width="400" height="300" fill="#f0f0f0"/><text x="200" y="150" font-family="Arial" font-size="20" fill="#999" text-anchor="middle" alignment-baseline="middle">No Image</text></svg>';
    
    // Get seller information
    const seller = getUserById(product.sellerId);
    
    card.innerHTML += `
        <img src="${mainImage}" alt="${product.title}" class="product-image" onerror="this.src='data:image/svg+xml;utf8,<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"400\" height=\"300\" viewBox=\"0 0 400 300\"><rect width=\"400\" height=\"300\" fill=\"#f0f0f0\"/><text x=\"200\" y=\"150\" font-family=\"Arial\" font-size=\"20\" fill=\"#999\" text-anchor=\"middle\" alignment-baseline=\"middle\">No Image</text></svg>'">
        <div class="product-info">
            <h3 class="product-title">${product.title}</h3>
            <p class="product-price">${formatCurrency(product.price)}</p>
            <div class="product-meta">
                <span>${getCategoryDisplayName(product.category)}</span>
                <span class="badge ${product.condition}">${getConditionDisplayName(product.condition)}</span>
            </div>
            <div class="product-meta">
                <span><i class="fas fa-map-marker-alt"></i> ${formatLocation(product.location)}</span>
                <span><i class="fas fa-user"></i> ${seller ? seller.name : product.sellerName}</span>
            </div>
            <div class="product-contact-buttons">
                <button class="btn btn-sm btn-primary contact-btn" data-product-id="${product.id}">
                    <i class="fas fa-comment"></i> Contact
                </button>
                ${seller && seller.contact ? 
                    `<button class="btn btn-sm btn-success whatsapp-btn" data-contact="${seller.contact}">
                        <i class="fab fa-whatsapp"></i> WhatsApp
                    </button>` : ''}
            </div>
        </div>
    `;
    
    // Add event listener to make card clickable (except for buttons)
    card.addEventListener('click', (e) => {
        // Don't navigate if clicking on buttons
        if (e.target.closest('.contact-btn') || e.target.closest('.whatsapp-btn')) {
            return;
        }
        // Store product in sessionStorage for detail page
        sessionStorage.setItem('currentProduct', JSON.stringify(product));
        window.location.href = `product-detail.html?id=${product.id}`;
    });
    
    // Add event listeners for contact buttons
    setTimeout(() => {
        const contactBtn = card.querySelector('.contact-btn');
        const whatsappBtn = card.querySelector('.whatsapp-btn');
        
        if (contactBtn) {
            contactBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                handleCardContact(product.id);
            });
        }
        
        if (whatsappBtn) {
            whatsappBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                const contact = whatsappBtn.getAttribute('data-contact');
                if (contact) {
                    const phoneNumber = contact.replace(/\D/g, '');
                    const whatsappUrl = `https://wa.me/${phoneNumber}`;
                    window.open(whatsappUrl, '_blank');
                    showNotification('Opening WhatsApp...', 'success');
                }
            });
        }
    }, 0);
    
    // Add owner options if needed
    if (showOwnerOptions && isProductOwner(product.id)) {
        const ownerActions = document.createElement('div');
        ownerActions.className = 'owner-actions';
        ownerActions.style.padding = '10px';
        ownerActions.style.borderTop = '1px solid #eee';
        ownerActions.style.textAlign = 'center';
        
        if (!product.isSold) {
            ownerActions.innerHTML = `
                <button class="btn btn-warning btn-sm mark-sold-btn" data-id="${product.id}">Mark as Sold</button>
                <button class="btn btn-danger btn-sm delete-product-btn" data-id="${product.id}">Delete</button>
            `;
        } else {
            ownerActions.innerHTML = `
                <button class="btn btn-secondary btn-sm" disabled>Sold</button>
            `;
        }
        
        // Add event listeners for owner actions
        setTimeout(() => {
            const markSoldBtn = ownerActions.querySelector('.mark-sold-btn');
            const deleteBtn = ownerActions.querySelector('.delete-product-btn');
            
            if (markSoldBtn) {
                markSoldBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    markProductAsSold(product.id);
                });
            }
            
            if (deleteBtn) {
                deleteBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    deleteProduct(product.id);
                });
            }
        }, 0);
        
        card.appendChild(ownerActions);
    }
    
    return card;
}

// Load product detail page
function loadProductDetail() {
    // Get product from URL or sessionStorage
    const urlParams = new URLSearchParams(window.location.search);
    const productId = urlParams.get('id');
    
    let product = null;
    
    // First try to get from URL parameter
    if (productId) {
        const products = loadProducts();
        product = products.find(p => p.id === productId);
    }
    
    // If not found in URL, try sessionStorage
    if (!product) {
        const storedProduct = sessionStorage.getItem('currentProduct');
        if (storedProduct) {
            product = JSON.parse(storedProduct);
        }
    }
    
    if (!product) {
        showNotification('Product not found', 'error');
        setTimeout(() => {
            window.location.href = 'index.html';
        }, 1500);
        return;
    }
    
    // Populate product details
    const mainImage = product.images && product.images.length > 0 ? product.images[0] : 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="400" height="300" viewBox="0 0 400 300"><rect width="400" height="300" fill="#f0f0f0"/><text x="200" y="150" font-family="Arial" font-size="20" fill="#999" text-anchor="middle" alignment-baseline="middle">No Image</text></svg>';
    document.getElementById('main-product-image').src = mainImage;
    document.getElementById('product-title').textContent = product.title;
    document.getElementById('product-price').textContent = formatCurrency(product.price);
    document.getElementById('product-description').textContent = product.description;
    
    // Get seller information
    const seller = getUserById(product.sellerId);
    if (seller) {
        document.getElementById('seller-shop-name').textContent = seller.shopName || 'No Shop Name';
        document.getElementById('seller-name').textContent = seller.name;
        document.getElementById('seller-contact').textContent = seller.contact || 'No contact';
        
        // Show location if available
        if (product.location) {
            document.getElementById('product-location-container').style.display = 'block';
            document.getElementById('product-location').textContent = formatLocation(product.location);
        }
    } else {
        document.getElementById('seller-shop-name').textContent = 'Seller Info Not Available';
        document.getElementById('seller-name').textContent = product.sellerName || 'Unknown';
        document.getElementById('seller-contact').textContent = 'Contact not available';
    }
    
    document.getElementById('product-date').textContent = formatDate(product.datePosted);
    
    // Set condition badge
    const conditionBadge = document.getElementById('product-condition-badge');
    conditionBadge.textContent = getConditionDisplayName(product.condition);
    conditionBadge.className = `badge ${product.condition}`;
    
    // Set sold badge if applicable
    const statusBadge = document.getElementById('product-status-badge');
    if (product.isSold) {
        statusBadge.style.display = 'inline';
    } else {
        statusBadge.style.display = 'none';
    }
    
    // Show seller options if current user is the owner
    const sellerOptions = document.getElementById('seller-options');
    if (isProductOwner(product.id)) {
        sellerOptions.style.display = 'block';
        
        // Update button text based on sold status
        const markSoldBtn = document.getElementById('mark-sold-btn');
        if (product.isSold) {
            markSoldBtn.textContent = 'Already Sold';
            markSoldBtn.disabled = true;
        } else {
            markSoldBtn.textContent = 'Mark as Sold';
            markSoldBtn.disabled = false;
        }
    } else {
        sellerOptions.style.display = 'none';
    }
    
    // Generate thumbnail images
    const thumbnailContainer = document.getElementById('thumbnail-images');
    if (thumbnailContainer && product.images && product.images.length > 1) {
        product.images.forEach((img, index) => {
            const thumb = document.createElement('img');
            thumb.src = img;
            thumb.alt = `Thumbnail ${index + 1}`;
            thumb.className = 'thumbnail-image';
            if (index === 0) thumb.classList.add('active');
            
            thumb.addEventListener('click', () => {
                document.getElementById('main-product-image').src = img;
                
                // Update active thumbnail
                document.querySelectorAll('.thumbnail-image').forEach(t => t.classList.remove('active'));
                thumb.classList.add('active');
            });
            
            // Add error handling for thumbnails
            thumb.onerror = function() {
                this.src = 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="100" height="100" viewBox="0 0 100 100"><rect width="100" height="100" fill="#f0f0f0"/><text x="50" y="50" font-family="Arial" font-size="10" fill="#999" text-anchor="middle" alignment-baseline="middle">No Img</text></svg>';
            };
            
            thumbnailContainer.appendChild(thumb);
        });
    }
}

// Handle contact seller
function handleContactSeller() {
    if (!isLoggedIn()) {
        showNotification('Please log in to contact sellers', 'warning');
        setTimeout(() => {
            window.location.href = 'login.html';
        }, 1500);
        return;
    }
    
    // Get seller contact info
    const urlParams = new URLSearchParams(window.location.search);
    const productId = urlParams.get('id');
    let product = null;
    
    if (productId) {
        const products = loadProducts();
        product = products.find(p => p.id === productId);
    }
    
    if (!product) {
        const storedProduct = sessionStorage.getItem('currentProduct');
        if (storedProduct) {
            product = JSON.parse(storedProduct);
        }
    }
    
    if (product) {
        const seller = getUserById(product.sellerId);
        if (seller && seller.contact) {
            showNotification(`Seller contact: ${seller.contact}`, 'success');
        } else {
            showNotification('Seller contact information not available', 'error');
        }
    }
}

// Handle WhatsApp contact
function handleWhatsAppContact() {
    if (!isLoggedIn()) {
        showNotification('Please log in to contact sellers', 'warning');
        setTimeout(() => {
            window.location.href = 'login.html';
        }, 1500);
        return;
    }
    
    // Get seller contact info
    const urlParams = new URLSearchParams(window.location.search);
    const productId = urlParams.get('id');
    let product = null;
    
    if (productId) {
        const products = loadProducts();
        product = products.find(p => p.id === productId);
    }
    
    if (!product) {
        const storedProduct = sessionStorage.getItem('currentProduct');
        if (storedProduct) {
            product = JSON.parse(storedProduct);
        }
    }
    
    if (product) {
        const seller = getUserById(product.sellerId);
        if (seller && seller.contact) {
            // Format phone number for WhatsApp (remove spaces, dashes, etc.)
            const phoneNumber = seller.contact.replace(/\D/g, '');
            const whatsappUrl = `https://wa.me/${phoneNumber}`;
            window.open(whatsappUrl, '_blank');
            showNotification('Opening WhatsApp...', 'success');
        } else {
            showNotification('WhatsApp contact not available', 'error');
        }
    }
}

// Handle contact from product card
function handleCardContact(productId) {
    if (!isLoggedIn()) {
        showNotification('Please log in to contact sellers', 'warning');
        setTimeout(() => {
            window.location.href = 'login.html';
        }, 1500);
        return;
    }
    
    const products = loadProducts();
    const product = products.find(p => p.id === productId);
    
    if (product) {
        const seller = getUserById(product.sellerId);
        if (seller && seller.contact) {
            showNotification(`Seller contact: ${seller.contact}`, 'success');
        } else {
            showNotification('Seller contact information not available', 'error');
        }
    }
}

// Handle call contact
function handleCallContact() {
    if (!isLoggedIn()) {
        showNotification('Please log in to contact sellers', 'warning');
        setTimeout(() => {
            window.location.href = 'login.html';
        }, 1500);
        return;
    }
    
    // Get seller contact info
    const urlParams = new URLSearchParams(window.location.search);
    const productId = urlParams.get('id');
    let product = null;
    
    if (productId) {
        const products = loadProducts();
        product = products.find(p => p.id === productId);
    }
    
    if (!product) {
        const storedProduct = sessionStorage.getItem('currentProduct');
        if (storedProduct) {
            product = JSON.parse(storedProduct);
        }
    }
    
    if (product) {
        const seller = getUserById(product.sellerId);
        if (seller && seller.contact) {
            // Format phone number for calling
            const phoneNumber = seller.contact.replace(/\D/g, '');
            window.location.href = `tel:${phoneNumber}`;
            showNotification('Initiating call...', 'success');
        } else {
            showNotification('Call contact not available', 'error');
        }
    }
}

// Mark product as sold
function handleMarkAsSold() {
    if (!requireAuth()) return;
    
    const urlParams = new URLSearchParams(window.location.search);
    const productId = urlParams.get('id');
    
    if (productId) {
        markProductAsSold(productId);
    }
}

// Mark product as sold (helper function)
function markProductAsSold(productId) {
    const products = loadProducts();
    const productIndex = products.findIndex(p => p.id === productId);
    
    if (productIndex === -1) {
        showNotification('Product not found', 'error');
        return;
    }
    
    const product = products[productIndex];
    
    if (product.sellerId !== getCurrentUser().id) {
        showNotification('You can only mark your own products as sold', 'error');
        return;
    }
    
    // Update product status
    products[productIndex].isSold = true;
    saveProducts(products);
    
    // Move product from posted to sold in user's profile
    const currentUser = getCurrentUser();
    const postedIndex = currentUser.postedProducts.indexOf(productId);
    if (postedIndex > -1) {
        currentUser.postedProducts.splice(postedIndex, 1);
        currentUser.soldProducts.push(productId);
        updateCurrentUser(currentUser);
        saveUser(currentUser);
    }
    
    showNotification('Product marked as sold!', 'success');
    
    // Reload the page to reflect changes
    setTimeout(() => {
        location.reload();
    }, 1000);
}

// Handle delete product
function handleDeleteProduct() {
    if (!requireAuth()) return;
    
    const urlParams = new URLSearchParams(window.location.search);
    const productId = urlParams.get('id');
    
    if (productId) {
        deleteProduct(productId);
    }
}

// Delete product (helper function)
function deleteProduct(productId) {
    if (!confirm('Are you sure you want to delete this product?')) {
        return;
    }
    
    const products = loadProducts();
    const productIndex = products.findIndex(p => p.id === productId);
    
    if (productIndex === -1) {
        showNotification('Product not found', 'error');
        return;
    }
    
    const product = products[productIndex];
    
    if (product.sellerId !== getCurrentUser().id) {
        showNotification('You can only delete your own products', 'error');
        return;
    }
    
    // Remove product from products array
    products.splice(productIndex, 1);
    saveProducts(products);
    
    // Remove product from user's posted/sold products
    const currentUser = getCurrentUser();
    const postedIndex = currentUser.postedProducts.indexOf(productId);
    const soldIndex = currentUser.soldProducts.indexOf(productId);
    
    if (postedIndex > -1) {
        currentUser.postedProducts.splice(postedIndex, 1);
    }
    if (soldIndex > -1) {
        currentUser.soldProducts.splice(soldIndex, 1);
    }
    
    updateCurrentUser(currentUser);
    saveUser(currentUser);
    
    showNotification('Product deleted successfully!', 'success');
    
    // Redirect back to profile
    setTimeout(() => {
        window.location.href = 'profile.html';
    }, 1000);
}

// Perform search
function performSearch() {
    const searchTerm = document.getElementById('search-input').value.toLowerCase();
    const products = loadProducts();
    
    const filteredProducts = products.filter(product => {
        return (
            product.title.toLowerCase().includes(searchTerm) ||
            product.description.toLowerCase().includes(searchTerm) ||
            product.location.toLowerCase().includes(searchTerm) ||
            getCategoryDisplayName(product.category).toLowerCase().includes(searchTerm)
        ) && !product.isSold;
    });
    
    displayFilteredProducts(filteredProducts);
}

// Apply filters
function applyFilters() {
    const categoryFilter = document.getElementById('category-filter').value;
    const locationFilter = document.getElementById('location-filter').value;
    const conditionFilter = document.getElementById('condition-filter').value;
    
    const products = loadProducts();
    
    const filteredProducts = products.filter(product => {
        const matchesCategory = !categoryFilter || product.category === categoryFilter;
        const matchesLocation = !locationFilter || product.location.toLowerCase().includes(locationFilter.toLowerCase());
        const matchesCondition = !conditionFilter || product.condition === conditionFilter;
        const isNotSold = !product.isSold;
        
        return matchesCategory && matchesLocation && matchesCondition && isNotSold;
    });
    
    displayFilteredProducts(filteredProducts);
}

// Display filtered products
function displayFilteredProducts(products) {
    const container = document.getElementById('products-grid');
    if (!container) return;
    
    container.innerHTML = '';
    
    if (products.length === 0) {
        container.innerHTML = `
            <div class="no-products-message">
                <h3>No products found</h3>
                <p>Try adjusting your search or filters</p>
            </div>
        `;
        return;
    }
    
    products.forEach(product => {
        const productCard = createProductCard(product);
        container.appendChild(productCard);
    });
}

// Populate location filter with unique locations
function populateLocationFilter() {
    const products = loadProducts();
    const locations = [...new Set(products.map(p => p.location))];
    
    const locationFilter = document.getElementById('location-filter');
    if (!locationFilter) return;
    
    // Clear existing options except the first one
    locationFilter.innerHTML = '<option value="">All Locations</option>';
    
    locations.forEach(location => {
        const option = document.createElement('option');
        option.value = location.toLowerCase();
        option.textContent = formatLocation(location);
        locationFilter.appendChild(option);
    });
}