// Authentication functions for Direct'sell marketplace

// Initialize authentication
function initAuth() {
    updateNavigation();
    setupEventListeners();
}

// Set up event listeners for auth-related elements
function setupEventListeners() {
    // Login form submission
    const loginForm = document.getElementById('login-form');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }

    // Registration form submission
    const registerForm = document.getElementById('register-form');
    if (registerForm) {
        registerForm.addEventListener('submit', handleRegistration);
    }

    // Logout button
    const logoutBtn = document.getElementById('logout-btn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', handleLogout);
    }

    // Tab switching for login/register
    const tabButtons = document.querySelectorAll('.tab-btn');
    tabButtons.forEach(button => {
        button.addEventListener('click', switchAuthTab);
    });

    // Navigation buttons
    const postProductBtn = document.getElementById('post-product-btn');
    const profileBtn = document.getElementById('profile-btn');
    const loginBtn = document.getElementById('login-btn');
    const postHeroBtn = document.getElementById('post-hero-btn');

    if (postProductBtn) {
        postProductBtn.addEventListener('click', (e) => {
            if (!isLoggedIn()) {
                e.preventDefault();
                window.location.href = 'login.html';
            }
        });
    }

    if (profileBtn) {
        profileBtn.addEventListener('click', (e) => {
            if (!isLoggedIn()) {
                e.preventDefault();
                window.location.href = 'login.html';
            }
        });
    }

    if (loginBtn) {
        loginBtn.addEventListener('click', (e) => {
            if (isLoggedIn()) {
                e.preventDefault();
                window.location.href = 'profile.html';
            }
        });
    }

    if (postHeroBtn) {
        postHeroBtn.addEventListener('click', (e) => {
            if (!isLoggedIn()) {
                e.preventDefault();
                window.location.href = 'login.html';
            } else {
                window.location.href = 'post-product.html';
            }
        });
    }
}

// Handle login form submission
function handleLogin(e) {
    e.preventDefault();

    const emailInput = document.getElementById('login-email');
    const passwordInput = document.getElementById('login-password');

    const email = emailInput.value.trim();
    const password = passwordInput.value;

    // Validation
    if (!email || !password) {
        showNotification('Please fill in all fields', 'error');
        return;
    }

    // Check if user exists
    const users = loadUsers();
    let user = null;

    // Check by email, phone, or contact
    for (let i = 0; i < users.length; i++) {
        if ((users[i].email && users[i].email.toLowerCase() === email.toLowerCase()) || 
            (users[i].phone && users[i].phone === email) ||
            (users[i].contact && users[i].contact === email)) {
            user = users[i];
            break;
        }
    }

    if (!user) {
        showNotification('User not found. Please register first.', 'error');
        return;
    }

    // Check password
    if (user.password !== password) {
        showNotification('Incorrect password', 'error');
        return;
    }

    // Successful login
    sessionStorage.setItem('currentUser', JSON.stringify(user));
    showNotification('Login successful!', 'success');
    
    // Update navigation
    updateNavigation();
    
    // Redirect to home or previous page
    setTimeout(() => {
        window.location.href = 'index.html';
    }, 1000);
}

// Handle registration form submission
function handleRegistration(e) {
    e.preventDefault();

    const shopNameInput = document.getElementById('reg-shop-name');
    const nameInput = document.getElementById('reg-name');
    const contactInput = document.getElementById('reg-contact');
    const emailInput = document.getElementById('reg-email');
    const passwordInput = document.getElementById('reg-password');
    const confirmPasswordInput = document.getElementById('reg-confirm-password');

    const shopName = shopNameInput.value.trim();
    const name = nameInput.value.trim();
    const contact = contactInput.value.trim();
    const email = emailInput.value.trim();
    const password = passwordInput.value;
    const confirmPassword = confirmPasswordInput.value;

    // Validation
    if (!shopName || !name || !contact || !password || !confirmPassword) {
        showNotification('Please fill in all required fields', 'error');
        return;
    }

    if (password !== confirmPassword) {
        showNotification('Passwords do not match', 'error');
        return;
    }

    if (password.length < 6) {
        showNotification('Password must be at least 6 characters', 'error');
        return;
    }

    if (!isValidPhone(contact)) {
        showNotification('Please enter a valid phone number', 'error');
        return;
    }

    // Check if email or phone is already registered
    const users = loadUsers();
    const existingUser = users.find(user => 
        (user.email && user.email.toLowerCase() === email.toLowerCase()) || 
        (user.phone && user.phone === contact) ||
        (user.contact && user.contact === contact)
    );

    if (existingUser) {
        showNotification('Account already exists with this contact information', 'error');
        return;
    }

    // Create new user
    const newUser = {
        id: generateId(),
        shopName: shopName,
        name: name,
        contact: contact,
        email: email ? email.toLowerCase() : null,
        phone: contact, // Store contact as phone
        password: password,
        postedProducts: [],
        soldProducts: [],
        location: '',
        createdAt: new Date().toISOString()
    };

    // Save user
    saveUser(newUser);

    // Log in the new user
    sessionStorage.setItem('currentUser', JSON.stringify(newUser));
    showNotification('Registration successful!', 'success');
    
    // Update navigation
    updateNavigation();
    
    // Redirect to home
    setTimeout(() => {
        window.location.href = 'index.html';
    }, 1000);
}

// Handle logout
function handleLogout(e) {
    e.preventDefault();
    sessionStorage.removeItem('currentUser');
    showNotification('Logged out successfully', 'success');
    
    // Update navigation
    updateNavigation();
    
    // Redirect to home
    window.location.href = 'index.html';
}

// Switch between login and register tabs
function switchAuthTab(e) {
    const tabName = e.target.getAttribute('data-tab');
    
    // Update active tab button
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    e.target.classList.add('active');
    
    // Show/hide forms
    document.querySelectorAll('.auth-tab-content').forEach(content => {
        content.classList.remove('active');
    });
    document.getElementById(`${tabName}-form`).classList.add('active');
}

// Check if user is logged in and redirect if needed
function requireAuth(redirectUrl = 'login.html') {
    if (!isLoggedIn()) {
        showNotification('Please log in to access this page', 'warning');
        setTimeout(() => {
            window.location.href = redirectUrl;
        }, 1500);
        return false;
    }
    return true;
}

// Check if user owns a product
function isProductOwner(productId) {
    if (!isLoggedIn()) return false;
    
    const currentUser = getCurrentUser();
    const products = loadProducts();
    const product = products.find(p => p.id === productId);
    
    return product && product.sellerId === currentUser.id;
}

// Load user profile data
function loadUserProfile() {
    if (!requireAuth()) return;
    
    const currentUser = getCurrentUser();
    
    const userShopNameElement = document.getElementById('user-shop-name');
    const userNameElement = document.getElementById('user-name');
    const userContactElement = document.getElementById('user-contact');
    const userEmailElement = document.getElementById('user-email');
    
    if (userShopNameElement) userShopNameElement.textContent = currentUser.shopName || 'No shop name';
    if (userNameElement) userNameElement.textContent = currentUser.name;
    if (userContactElement) userContactElement.textContent = currentUser.contact ? `📞 ${currentUser.contact}` : 'No contact info';
    if (userEmailElement) userEmailElement.textContent = currentUser.email ? `✉️ ${currentUser.email}` : 'No email';
    
    // Load user's posted products
    loadUserPostedProducts();
    
    // Load user's sold products
    loadUserSoldProducts();
}

// Load user's posted products
function loadUserPostedProducts() {
    if (!requireAuth()) return;
    
    const currentUser = getCurrentUser();
    const products = loadProducts();
    
    // Filter products posted by current user that are not sold
    const postedProducts = products.filter(product => 
        product.sellerId === currentUser.id && !product.isSold
    );
    
    const container = document.getElementById('posted-products-grid');
    if (!container) return;
    
    container.innerHTML = '';
    
    if (postedProducts.length === 0) {
        container.innerHTML = '<p class="no-products">You have not posted any products yet.</p>';
        return;
    }
    
    postedProducts.forEach(product => {
        const productCard = createProductCard(product, true); // Show owner options
        container.appendChild(productCard);
    });
}

// Load user's sold products
function loadUserSoldProducts() {
    if (!requireAuth()) return;
    
    const currentUser = getCurrentUser();
    const products = loadProducts();
    
    // Filter products posted by current user that are sold
    const soldProducts = products.filter(product => 
        product.sellerId === currentUser.id && product.isSold
    );
    
    const container = document.getElementById('sold-products-grid');
    if (!container) return;
    
    container.innerHTML = '';
    
    if (soldProducts.length === 0) {
        container.innerHTML = '<p class="no-products">You have not sold any products yet.</p>';
        return;
    }
    
    soldProducts.forEach(product => {
        const productCard = createProductCard(product, true); // Show owner options
        container.appendChild(productCard);
    });
}

// Switch between profile tabs
function setupProfileTabs() {
    const tabButtons = document.querySelectorAll('.profile-tabs .tab-btn');
    tabButtons.forEach(button => {
        button.addEventListener('click', (e) => {
            const tabName = e.target.getAttribute('data-tab');
            
            // Update active tab button
            document.querySelectorAll('.profile-tabs .tab-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            e.target.classList.add('active');
            
            // Show/hide content
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            document.getElementById(`${tabName}-products`).classList.add('active');
        });
    });
    
    // Setup edit profile functionality
    setupEditProfile();
}

// Setup edit profile functionality
function setupEditProfile() {
    const editBtn = document.getElementById('edit-profile-btn');
    const editForm = document.getElementById('edit-profile-form');
    const cancelBtn = document.getElementById('cancel-edit-btn');
    const profileEditForm = document.getElementById('profile-edit-form');
    
    if (editBtn) {
        editBtn.addEventListener('click', () => {
            // Show edit form and populate with current data
            if (editForm) {
                editForm.style.display = 'block';
                populateEditForm();
            }
        });
    }
    
    if (cancelBtn) {
        cancelBtn.addEventListener('click', () => {
            if (editForm) {
                editForm.style.display = 'none';
            }
        });
    }
    
    if (profileEditForm) {
        profileEditForm.addEventListener('submit', handleProfileEdit);
    }
}

// Populate edit form with current user data
function populateEditForm() {
    if (!requireAuth()) return;
    
    const currentUser = getCurrentUser();
    
    const shopNameInput = document.getElementById('edit-shop-name');
    const nameInput = document.getElementById('edit-name');
    const contactInput = document.getElementById('edit-contact');
    const emailInput = document.getElementById('edit-email');
    const locationInput = document.getElementById('edit-location');
    
    if (shopNameInput) shopNameInput.value = currentUser.shopName || '';
    if (nameInput) nameInput.value = currentUser.name || '';
    if (contactInput) contactInput.value = currentUser.contact || '';
    if (emailInput) emailInput.value = currentUser.email || '';
    if (locationInput) locationInput.value = currentUser.location || '';
}

// Handle profile edit form submission
function handleProfileEdit(e) {
    e.preventDefault();
    
    if (!requireAuth()) return;
    
    const shopNameInput = document.getElementById('edit-shop-name');
    const nameInput = document.getElementById('edit-name');
    const contactInput = document.getElementById('edit-contact');
    const emailInput = document.getElementById('edit-email');
    const locationInput = document.getElementById('edit-location');
    
    const shopName = shopNameInput.value.trim();
    const name = nameInput.value.trim();
    const contact = contactInput.value.trim();
    const email = emailInput.value.trim();
    const location = locationInput.value.trim();
    
    // Validation
    if (!shopName || !name || !contact) {
        showNotification('Please fill in all required fields', 'error');
        return;
    }
    
    if (!isValidPhone(contact)) {
        showNotification('Please enter a valid phone number', 'error');
        return;
    }
    
    if (email && !isValidEmail(email)) {
        showNotification('Please enter a valid email address', 'error');
        return;
    }
    
    // Update user data
    const currentUser = getCurrentUser();
    currentUser.shopName = shopName;
    currentUser.name = name;
    currentUser.contact = contact;
    currentUser.email = email ? email.toLowerCase() : null;
    currentUser.location = location;
    
    // Save updated user
    saveUser(currentUser);
    updateCurrentUser(currentUser);
    
    // Update display
    loadUserProfile();
    
    // Hide edit form
    const editForm = document.getElementById('edit-profile-form');
    if (editForm) {
        editForm.style.display = 'none';
    }
    
    showNotification('Profile updated successfully!', 'success');
}